package gov.va.vamf.scheduling.varutility.domain;

import agilex.commons.NullChecker;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.*;

import static gov.va.vamf.scheduling.varutility.utils.VarUtilityConstants.REQUEST_SCHEDULING_METHOD;

@XmlAccessorType(value = XmlAccessType.NONE)
@XmlRootElement(namespace = Namespace.VarUtility)
public class CoreSettings extends DomainBaseObjects<CoreSetting> {

    public static final String PRIMARY_CARE_ID = "323";

    String[] booleans   = {"Yes", "No"};
    Integer[] durations = {365, 730};
    ArrayList<String> careTypeIdsList = new ArrayList<String>();
    ArrayList<String> validBooleans = new ArrayList<String>(Arrays.asList(booleans));
    ArrayList<Integer> validDurations = new ArrayList<Integer>(Arrays.asList(durations));

    public ArrayList<ValidationError> validate(List<ClinicalServices> clinicalServices, String schedulingMethod) {
        ArrayList<ValidationError> errors = new ArrayList<ValidationError>();

        validateTypesOfCare(errors, clinicalServices);
        validatePatientHistoryRequired(errors);
        validateDurations(errors);

        if (schedulingMethod.equals(REQUEST_SCHEDULING_METHOD)) {
            validateSubmittedRequestLimit(errors, clinicalServices);
        } else {
            validateCanCancel(errors);
        }
        return errors;
    }

    private void validateDurations(ArrayList<ValidationError> errors) {
        for(CoreSetting c : this) {
            if(NullChecker.isNotNullish(c.getPatientHistoryRequired()) &&
                c.getPatientHistoryRequired().equals("Yes") &&
                NullChecker.isNullish(c.getPatientHistoryDuration())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "PatientHistoryDuration is required."));
            }
            if(NullChecker.isNotNullish(c.getPatientHistoryDuration()) &&
                ! validDurations.contains(c.getPatientHistoryDuration())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "Valid values for patientHistoryDuration are 365, 730, or null"));
            }
        }
    }

    private void validatePatientHistoryRequired(ArrayList<ValidationError> errors) {
        for(CoreSetting c : this) {
            if(NullChecker.isNotNullish(c.getPatientHistoryRequired()) &&
                ! validBooleans.contains(c.getPatientHistoryRequired())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "Valid values for patientHistoryRequired are Yes, No, or null"));
            }
        }
    }

    private void validateTypesOfCare(ArrayList<ValidationError> errors, List<ClinicalServices> clinicalServices) {
        for (ClinicalServices clinicalService : clinicalServices) {
            careTypeIdsList.add(clinicalService.getId());
        }
        ArrayList<String> inputCareTypeIdsList = new ArrayList<String>();

        for(CoreSetting c : this) {
            inputCareTypeIdsList.add(c.getId());
        }

        if(! inputCareTypeIdsList.containsAll(careTypeIdsList)) {
            errors.add(new ValidationError("CoreSettings does not contain all care types."));
        }

        if(inputCareTypeIdsList.size() != careTypeIdsList.size()) {
            errors.add(new ValidationError("Invalid care types."));
        }
    }

    private void validateCanCancel(ArrayList<ValidationError> errors) {
        for(CoreSetting c : this) {
            if(c.getCanCancel() == null) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                        "Cancel appointment is required."));
            }
        }
    }

    private void validateSubmittedRequestLimit(ArrayList<ValidationError> errors, List<ClinicalServices> clinicalServices) {
        Map<String, Integer> serviceSubmittedRequestLimits = new HashMap<String, Integer>();
        for (ClinicalServices clinicalService : clinicalServices) {
            serviceSubmittedRequestLimits.put(clinicalService.getId(), clinicalService.getSubmittedRequestLimit());
        }
        for(CoreSetting c : this) {
            if (c.getSubmittedRequestLimit() == null) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                        "Submitted Request Limit is required"));
            } else {
                if (c.getSubmittedRequestLimit() > serviceSubmittedRequestLimits.get(c.getId())) {
                    errors.add(new ValidationError(c.getTypeOfCare(), "Submitted Request Limit cannot be more than " + serviceSubmittedRequestLimits.get(c.getId())));
                }
            }
        }
    }

    public void cleanupDataForSave() {
        for(CoreSetting cs : this) {
            if(cs.getId().equals(PRIMARY_CARE_ID)) {
                cs.setPatientHistoryDuration(null);
            } else {
                if(NullChecker.isNullish(cs.getPatientHistoryRequired()) || cs.getPatientHistoryRequired().equals("No")) {
                    cs.setPatientHistoryDuration(0);
                }
            }
            cs.setSubmittedRequestLimit(null);
        }
    }
}
